import 'package:aad_oauth/aad_oauth.dart';
import 'package:absenkuv3/Bloc/AuthBloc.dart';
import 'package:absenkuv3/Bloc/NavigatorBloc.dart';
import 'package:absenkuv3/Setting/oauth_config.dart';
import 'package:absenkuv3/UI/Template/style.dart';
import 'package:absenkuv3/UI/Template/templateButton.dart';
import 'package:flutter/material.dart';
import 'package:flutter_bloc/flutter_bloc.dart';
import 'package:absenkuv3/Bloc/SplashScreenBloc.dart';
import 'package:absenkuv3/Setting/value_image.dart';

class HalamanSplashScreen extends StatefulWidget {
  const HalamanSplashScreen({super.key});

  @override
  State<HalamanSplashScreen> createState() => _HalamanSplashScreenState();
}

class _HalamanSplashScreenState extends State<HalamanSplashScreen>
    with TickerProviderStateMixin {
  final GlobalKey<ScaffoldState> _scaffoldKey = GlobalKey<ScaffoldState>();

  late AnimationController _cardController;
  late Animation<Offset> _cardSlideAnimation;
  late Animation<double> _cardFadeAnimation;
  late Animation<double> _cardScaleAnimation;

  late AnimationController _logoController;
  late Animation<Offset> _logoSlideAnimation;
  late Animation<double> _logoFadeAnimation;
  late Animation<double> _logoScaleAnimation;

  late AnimationController _bgController;
  late Animation<double> _bgSlideAnimation;

  bool _allowAnimation = true;
  late AadOAuth oauth;

  @override
  void initState() {
    super.initState();
    oauth = OAuthService.oauth;
    BlocProvider.of<SplashScreenBloc>(context).add(SplashStarted(context));

    // Background animation controller
    _bgController = AnimationController(
      vsync: this,
      duration: const Duration(milliseconds: 1500),
    );

    _bgSlideAnimation = Tween<double>(
      begin: 0,
      end: -0.4,
    ).animate(CurvedAnimation(
      parent: _bgController,
      curve: Curves.easeInOutQuint,
    ));

    // Card animation controller
    _cardController = AnimationController(
      vsync: this,
      duration: const Duration(milliseconds: 1200),
    );

    _cardSlideAnimation = Tween<Offset>(
      begin: const Offset(0, 1.5),
      end: Offset.zero,
    ).animate(CurvedAnimation(
      parent: _cardController,
      curve: Curves.elasticOut,
    ));

    _cardFadeAnimation = Tween<double>(
      begin: 0,
      end: 1,
    ).animate(CurvedAnimation(
      parent: _cardController,
      curve: const Interval(0.3, 1.0, curve: Curves.easeInOutCubic),
    ));

    _cardScaleAnimation = Tween<double>(
      begin: 0.9,
      end: 1.0,
    ).animate(CurvedAnimation(
      parent: _cardController,
      curve: const Interval(0.6, 1.0, curve: Curves.easeOutBack),
    ));

    // Logo animation controller
    _logoController = AnimationController(
      vsync: this,
      duration: const Duration(milliseconds: 1000),
    );

    _logoSlideAnimation = Tween<Offset>(
      begin: Offset.zero,
      end: const Offset(0, -0.8),
    ).animate(CurvedAnimation(
      parent: _logoController,
      curve: Curves.easeInOutCubic,
    ));

    _logoFadeAnimation = Tween<double>(
      begin: 1,
      end: 0.2,
    ).animate(CurvedAnimation(
      parent: _logoController,
      curve: Curves.easeOut,
    ));

    _logoScaleAnimation = Tween<double>(
      begin: 1.0,
      end: 0.6,
    ).animate(CurvedAnimation(
      parent: _logoController,
      curve: Curves.easeInOut,
    ));
  }

  void startAnimation() {
    Future.delayed(const Duration(milliseconds: 500), () {
      if (_allowAnimation) {
        _bgController.forward();
        _cardController.forward();
        Future.delayed(const Duration(milliseconds: 300), () async {
          await _logoController.forward();
          setState(() {});
        });
      }
    });
  }

  void _loginAndFetchUser() async {
    try {
      await oauth.login();
      final accessToken = await oauth.getAccessToken();
      print('Access Token: $accessToken');

      if (accessToken != null && accessToken.isNotEmpty) {
        BlocProvider.of<AuthBloc>(context)
            .add(LoginSSOEvent(accessToken, context, _scaffoldKey));
      } else {
        ScaffoldMessenger.of(context).showSnackBar(
          const SnackBar(content: Text('Gagal mendapatkan access token.')),
        );
      }
    } catch (e) {
      print('Login error: $e');
      ScaffoldMessenger.of(context).showSnackBar(
        SnackBar(content: Text('Login error: $e')),
      );
    }
  }

  @override
  void dispose() {
    _cardController.dispose();
    _logoController.dispose();
    _bgController.dispose();
    super.dispose();
  }

  @override
  Widget build(BuildContext context) {
    final size = MediaQuery.of(context).size;

    return BlocListener<SplashScreenBloc, SplashState>(
      listener: (context, state) {
        if (state is SplashToDashboard) {
          _allowAnimation = false;
          BlocProvider.of<NavigatorBloc>(context)
              .add(NavigateRemoveToDashboard(context));
        } else if (state is SplashToLogin) {
          _allowAnimation = true;
          startAnimation();
        }
      },
      child: Scaffold(
        key: _scaffoldKey,
        body: Stack(
          children: [
            // Background with parallax effect
            AnimatedBuilder(
              animation: _bgController,
              builder: (context, child) {
                return Transform.translate(
                  offset: Offset(0, size.height * _bgSlideAnimation.value),
                  child: SizedBox(
                    width: size.width,
                    height: size.height,
                    child: Image.asset(
                      ValueImage.bgSplashScreen,
                      fit: BoxFit.cover,
                    ),
                  ),
                );
              },
            ),

            // Main logo with combined animations
            AnimatedBuilder(
              animation: Listenable.merge([_logoController, _cardController]),
              builder: (context, child) {
                const logoSize = 300.0;
                
                // Calculate combined animation effects
                final cardProgress = _cardController.value;
                final logoProgress = _logoController.value;
                
                // When card is appearing, logo moves up faster
                final verticalOffset = -0.5 * cardProgress + (-0.8 * logoProgress);
                
                // Combine opacity effects
                final combinedOpacity = (1 - (cardProgress * 0.7)) * _logoFadeAnimation.value;
                
                // Combine scale effects
                final combinedScale = (0.8 + (0.2 * (1 - cardProgress))) * _logoScaleAnimation.value;

                return IgnorePointer(
                  child: Align(
                    alignment: Alignment.center,
                    child: Transform.translate(
                      offset: Offset(0, verticalOffset * size.height),
                      child: Opacity(
                        opacity: combinedOpacity,
                        child: Transform.scale(
                          scale: combinedScale,
                          child: Container(
                            width: logoSize,
                            height: logoSize,
                            padding: const EdgeInsets.all(8),
                            child: Image.asset(
                              ValueImage.logoSplashScreen,
                              fit: BoxFit.contain,
                            ),
                          ),
                        ),
                      ),
                    ),
                  ),
                );
              },
            ),

            // Login card with multiple animations
            Align(
              alignment: Alignment.bottomCenter,
              child: AnimatedBuilder(
                animation: _cardController,
                builder: (context, child) {
                  return SlideTransition(
                    position: _cardSlideAnimation,
                    child: FadeTransition(
                      opacity: _cardFadeAnimation,
                      child: ScaleTransition(
                        scale: _cardScaleAnimation,
                        child: Container(
                          width: double.infinity,
                          height: size.height * 0.5,
                          decoration: BoxDecoration(
                            color: Colors.white,
                            borderRadius: const BorderRadius.vertical(
                                top: Radius.circular(30)),
                            boxShadow: [
                              BoxShadow(
                                color: Colors.black.withOpacity(0.3),
                                blurRadius: 25,
                                spreadRadius: 5,
                                offset: const Offset(0, -5),
                              ),
                            ],
                          ),
                          child: SingleChildScrollView(
                            child: Column(
                              crossAxisAlignment: CrossAxisAlignment.stretch,
                              children: [
                                Column(
                                  children: [
                                    const SizedBox(height: 30),
                                    Center(
                                      child: Image.asset(
                                        ValueImage.logoCardLogin,
                                        width: 300,
                                        height: 150,
                                        fit: BoxFit.contain,
                                      ),
                                    ),
                                    const SizedBox(height: 30),
                                    AnimatedOpacity(
                                      opacity: _cardController.value,
                                      duration: const Duration(milliseconds: 500),
                                      child: Column(
                                        children: [
                                          Text(
                                            "Selamat Datang",
                                            style: TextStyle(
                                              fontWeight: FontWeight.bold,
                                              fontSize: 25,
                                              color: Colors.black87,
                                            ),
                                          ),
                                          const SizedBox(height: 10),
                                          Text(
                                            "Login Akunmu menggunakan Azure",
                                            style: TextStyle(
                                              fontSize: 18,
                                              color: Colors.black54,
                                            ),
                                          ),
                                        ],
                                      ),
                                    ),
                                    const SizedBox(height: 20),
                                    Padding(
                                      padding: const EdgeInsets.symmetric(
                                          horizontal: 25),
                                      child: AnimatedOpacity(
                                        opacity: _cardController.value,
                                        duration: const Duration(milliseconds: 500),
                                        child: TemplateButton(
                                          onClick: _loginAndFetchUser,
                                          label: "Login Azure",
                                          color: CustomStyle.color1,
                                          fontSize: CustomStyle.fontSizeNormal,
                                          paddingTopBottom: 12,
                                          expand: true,
                                          borderColor: Colors.white,
                                          radius: 24,
                                          heightButton: 50,
                                        ),
                                      ),
                                    ),
                                  ],
                                ),
                              ],
                            ),
                          ),
                        ),
                      ),
                    ),
                  );
                },
              ),
            ),
          ],
        ),
      ),
    );
  }
}